
%% Compares DAMAS, quadprog, lsqnonneg and Lawson-Hanson on experimental data
%
% 2D grid of 10 800 points
% plots the objective function in function of time
% and power maps

% Approx 10min.

% loads the data (cov. matrix, frequency, array coordinates)
load damasdemo

% we select the 64 inner microphones
N = Pmic(:, 2).^2 + Pmic(:, 1).^2;
[~, order] = sort(N);
Nm = 64;
Z = order(1:Nm);
Pmic = Pmic(Z, :);
Data = Data(Z, Z);

close all

% source grid
Lx = 180;
Ly = 60;
Lz = 1;
xx = linspace(-2, 1, Lx)';
yy = linspace(-1, 0, Ly)';
zz = 4.4;
[Xg, Yg, Zg] = meshgrid(xx, yy, zz);

% dictionary of sources
D = dictionary(Pmic, [Xg(:) Yg(:) Zg(:)], k);

%% Beamforming (no normalization)
% used as input for DAMAS
Cbf = sum(conj(D) .* (Data*D), 1);
Cbf = real(Cbf');

%% Beamforming (normalized)
% used to plot the power map
Dbf = D ./ sum(abs(D).^2, 1);
Cbfn = sum(conj(Dbf) .* (Data*Dbf), 1);
Cbfn = real(Cbfn');

%% Large matrix for DAMAS
DD = abs(D'*D).^2;


%% Quadprog
tic;
xquad = quadprog(DD, -real(Cbf), [], [], [], [], zeros(Lx*Ly, 1), []);
% Time
Tquad = toc;
% Objective
Cquad = norm(D*diag(xquad)*D' - Data, 'fro')^2;

%% Damas
Niter = [10 20 50 100 200];

Tdamas = zeros(size(Niter));
Cd = zeros(size(Niter));

for u = 1:length(Niter)
    tic;
    xdamas = damas(DD, real(Cbf), Niter(u));
    Tdamas(u) = toc;
    Cdamas(u) = norm(D*diag(xdamas)*D' - Data, 'fro')^2;
end
%% lsqnonneg

M = size(D, 2);
N = size(D, 1);

% We prepare the matrix Dtilde (lsqnonneg does not take complex matrices)
DDDri = zeros(2*N^2, M);

for u = 1:M
    d = D(:, u)*D(:, u)';

    DDDri(:, u) = [real(d(:)) ; imag(d(:))];
end

Xri = [real(Data(:)) ; imag(Data(:))];

tic;
xlsqnonneg = lsqnonneg(DDDri, Xri);
Tlsqnonneg = toc;
Clsqnonneg = norm(D*diag(xlsqnonneg)*D' - Data, 'fro')^2;

%% optimized NNLS

tic;
xlh = cmf_nnls(D, Data, 1e2);
Tlh = toc;

Clh = norm(D*spdiags(xlh,0,size(D, 2), size(D,2))*D' - Data, 'fro')^2;

clear DD DDDri
save 2Dexp
%% Plots

% Objective function in function of computational time
figure
set(gcf, 'Position',  [100, 100, 500, 400])
scatter(Tquad, Cquad, 100, 'black', 'o', 'filled')
hold on
scatter(Tlsqnonneg, Clsqnonneg, 100, 'black', '+', 'linewidth', 2)
scatter(Tlh, Clh, 100, 'black', 'x', 'linewidth', 2)

plot(Tdamas, Cdamas, 'k-+', 'linewidth', 2)

for u = 1:length(Niter)
    text(Tdamas(u)+5, Cdamas(u)+2e10, sprintf("%u", Niter(u)))
end

xlabel('Time (s)')
ylabel('Objective function')

legend('quadprog', 'lsqnonneg', 'Lawson-Hanson (optimized)', 'DAMAS')
ylim([1.04e14 1.05e14])
grid on



%% Maps: DAMAS
m = 40;
M = max(10*log10(Cbfn));

figure
set(gcf, 'Position',  [100, 100, 500, 200])

imagesc(xx,yy,reshape(10*log10(xdamas), Ly, Lx))
axis xy
axis image
colormap((hot))


xlabel("X")
ylabel("Y")
ax = gca;
ax.CLim =[m M];

colorbar
title("DAMAS")
%% Maps: CMF-NNLS

figure
set(gcf, 'Position',  [100, 100, 500, 200])

imagesc(xx,yy,reshape(10*log10(xlh), Ly, Lx))
axis xy
axis image
colormap((hot))

xlabel("X")
ylabel("Y")
ax = gca;
ax.CLim =[m M];
colorbar
title("CMF-NNLS")
%% Maps: Beamforming

figure
set(gcf, 'Position',  [100, 100, 500, 200])

imagesc(xx,yy,reshape(10*log10(Cbfn), Ly, Lx))
axis xy
axis image
colormap((hot))

colorbar
ax = gca;
ax.CLim =[m M];
xlabel("X")
ylabel("Y")

title("Beamforming")